<?php
/**
 * Comprehensive Payroll Debugging Script
 * This script diagnoses all potential issues with the payroll module
 */

// Load WordPress
require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Access denied. Admin only.');
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>Payroll Module Diagnostics</title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; padding: 40px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #2c3e50; border-bottom: 3px solid #3498db; padding-bottom: 10px; }
        h2 { color: #34495e; margin-top: 30px; background: #ecf0f1; padding: 10px; border-left: 4px solid #3498db; }
        .success { background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 12px; margin: 10px 0; border-radius: 4px; }
        .error { background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 12px; margin: 10px 0; border-radius: 4px; }
        .warning { background: #fff3cd; border: 1px solid #ffeaa7; color: #856404; padding: 12px; margin: 10px 0; border-radius: 4px; }
        .info { background: #d1ecf1; border: 1px solid #bee5eb; color: #0c5460; padding: 12px; margin: 10px 0; border-radius: 4px; }
        table { width: 100%; border-collapse: collapse; margin: 15px 0; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background: #3498db; color: white; font-weight: 600; }
        tr:hover { background: #f8f9fa; }
        code { background: #f4f4f4; padding: 2px 6px; border-radius: 3px; font-family: 'Courier New', monospace; }
        .badge { display: inline-block; padding: 4px 8px; border-radius: 3px; font-size: 12px; font-weight: 600; }
        .badge-success { background: #28a745; color: white; }
        .badge-danger { background: #dc3545; color: white; }
        .badge-warning { background: #ffc107; color: #000; }
    </style>
</head>
<body>
<div class="container">
    <h1>🔍 Payroll Module Complete Diagnostics</h1>
    <p><strong>Timestamp:</strong> <?php echo date('Y-m-d H:i:s'); ?></p>

    <?php
    global $wpdb;
    $issues_found = [];
    $fixes_applied = [];

    // ========================================
    // 1. CHECK DATABASE TABLES
    // ========================================
    echo "<h2>1. Database Tables Check</h2>";
    
    $required_tables = [
        'hrm_payroll' => 'Payroll records',
        'hrm_salary_structures' => 'Employee salary structures',
        'hrm_attendance' => 'Attendance data (for calculations)'
    ];

    foreach ($required_tables as $table => $description) {
        $table_name = $wpdb->prefix . $table;
        $exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name;
        
        if ($exists) {
            $count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
            echo "<div class='success'>✓ Table <code>$table_name</code> exists ($description) - <strong>$count records</strong></div>";
        } else {
            echo "<div class='error'>✗ Table <code>$table_name</code> MISSING ($description)</div>";
            $issues_found[] = "Missing table: $table_name";
        }
    }

    // ========================================
    // 2. CHECK TABLE STRUCTURE
    // ========================================
    echo "<h2>2. Table Structure Validation</h2>";
    
    $payroll_table = $wpdb->prefix . 'hrm_payroll';
    if ($wpdb->get_var("SHOW TABLES LIKE '$payroll_table'") === $payroll_table) {
        $columns = $wpdb->get_results("DESCRIBE $payroll_table");
        echo "<table>";
        echo "<tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
        foreach ($columns as $col) {
            echo "<tr><td><code>{$col->Field}</code></td><td>{$col->Type}</td><td>{$col->Null}</td><td>{$col->Key}</td><td>{$col->Default}</td></tr>";
        }
        echo "</table>";
    }

    // ========================================
    // 3. CHECK PAYROLL DATA
    // ========================================
    echo "<h2>3. Payroll Data Analysis</h2>";
    
    $total_payroll = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}hrm_payroll");
    $draft_payroll = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}hrm_payroll WHERE status = 'draft'");
    $approved_payroll = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}hrm_payroll WHERE status = 'approved'");
    
    echo "<div class='info'>";
    echo "<strong>Total Payroll Records:</strong> $total_payroll<br>";
    echo "<strong>Draft Records:</strong> $draft_payroll<br>";
    echo "<strong>Approved Records:</strong> $approved_payroll";
    echo "</div>";

    if ($total_payroll == 0) {
        echo "<div class='warning'>⚠ No payroll records found. This is why the payroll section appears empty.</div>";
        $issues_found[] = "No payroll data exists";
    }

    // Show recent payroll records
    $recent_payroll = $wpdb->get_results("
        SELECT pr.*, p.post_title as employee_name 
        FROM {$wpdb->prefix}hrm_payroll pr
        LEFT JOIN {$wpdb->posts} p ON pr.employee_id = p.ID
        ORDER BY pr.created_at DESC
        LIMIT 10
    ");

    if ($recent_payroll) {
        echo "<h3>Recent Payroll Records (Last 10)</h3>";
        echo "<table>";
        echo "<tr><th>ID</th><th>Employee</th><th>Month</th><th>Net Salary</th><th>Status</th><th>Created</th></tr>";
        foreach ($recent_payroll as $pr) {
            $status_class = $pr->status === 'approved' ? 'success' : 'warning';
            echo "<tr>";
            echo "<td>#{$pr->id}</td>";
            echo "<td>{$pr->employee_name}</td>";
            echo "<td>{$pr->payroll_month}</td>";
            echo "<td>\${$pr->net_salary}</td>";
            echo "<td><span class='badge badge-{$status_class}'>{$pr->status}</span></td>";
            echo "<td>{$pr->created_at}</td>";
            echo "</tr>";
        }
        echo "</table>";
    }

    // ========================================
    // 4. CHECK SALARY STRUCTURES
    // ========================================
    echo "<h2>4. Salary Structures Check</h2>";
    
    $total_employees = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish'");
    $employees_with_salary = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}hrm_salary_structures");
    
    echo "<div class='info'>";
    echo "<strong>Total Active Employees:</strong> $total_employees<br>";
    echo "<strong>Employees with Salary Structure:</strong> $employees_with_salary";
    echo "</div>";

    if ($employees_with_salary == 0 && $total_employees > 0) {
        echo "<div class='warning'>⚠ No salary structures defined. Employees need salary structures before payroll can be generated.</div>";
        $issues_found[] = "No salary structures configured";
    }

    // Show employees without salary structure
    $employees_no_salary = $wpdb->get_results("
        SELECT p.ID, p.post_title 
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->prefix}hrm_salary_structures s ON p.ID = s.employee_id
        WHERE p.post_type = 'hrm_employee' 
        AND p.post_status = 'publish'
        AND s.id IS NULL
        LIMIT 10
    ");

    if ($employees_no_salary) {
        echo "<div class='warning'>";
        echo "<strong>Employees without salary structure:</strong><br>";
        foreach ($employees_no_salary as $emp) {
            echo "• {$emp->post_title} (ID: {$emp->ID})<br>";
        }
        echo "</div>";
    }

    // ========================================
    // 5. CHECK AJAX HANDLERS
    // ========================================
    echo "<h2>5. AJAX Handlers Registration</h2>";
    
    $ajax_actions = [
        'hrm_save_salary_structure',
        'hrm_generate_payroll',
        'hrm_update_payroll_status',
        'hrm_get_payslip',
        'hrm_delete_payroll',
        'hrm_save_payroll_settings'
    ];

    foreach ($ajax_actions as $action) {
        $has_action = has_action("wp_ajax_$action");
        if ($has_action) {
            echo "<div class='success'>✓ AJAX handler <code>$action</code> is registered</div>";
        } else {
            echo "<div class='error'>✗ AJAX handler <code>$action</code> NOT registered</div>";
            $issues_found[] = "Missing AJAX handler: $action";
        }
    }

    // ========================================
    // 6. CHECK CLASS AND METHODS
    // ========================================
    echo "<h2>6. Class and Methods Verification</h2>";
    
    if (class_exists('HRM_Payroll')) {
        echo "<div class='success'>✓ HRM_Payroll class exists</div>";
        
        $methods = [
            'init',
            'render_payroll',
            'render_dashboard',
            'render_salary_structures',
            'render_generate_payroll',
            'render_payroll_history',
            'handle_save_salary_structure',
            'handle_generate_payroll',
            'handle_get_payslip'
        ];
        
        foreach ($methods as $method) {
            if (method_exists('HRM_Payroll', $method)) {
                echo "<div class='success'>✓ Method <code>HRM_Payroll::$method()</code> exists</div>";
            } else {
                echo "<div class='error'>✗ Method <code>HRM_Payroll::$method()</code> MISSING</div>";
                $issues_found[] = "Missing method: HRM_Payroll::$method()";
            }
        }
    } else {
        echo "<div class='error'>✗ HRM_Payroll class NOT FOUND</div>";
        $issues_found[] = "HRM_Payroll class missing";
    }

    // ========================================
    // 7. CHECK ATTENDANCE DATA
    // ========================================
    echo "<h2>7. Attendance Data (Required for Payroll Calculation)</h2>";
    
    $current_month = date('Y-m');
    $first_day = $current_month . '-01';
    $last_day = date('Y-m-t', strtotime($first_day));
    
    $attendance_count = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*) FROM {$wpdb->prefix}hrm_attendance 
        WHERE date BETWEEN %s AND %s
    ", $first_day, $last_day));
    
    echo "<div class='info'>";
    echo "<strong>Attendance records for current month ($current_month):</strong> $attendance_count";
    echo "</div>";

    if ($attendance_count == 0) {
        echo "<div class='warning'>⚠ No attendance data for current month. Payroll calculations may be affected.</div>";
    }

    // ========================================
    // 8. TEST SQL QUERIES
    // ========================================
    echo "<h2>8. SQL Query Testing</h2>";
    
    // Test the dashboard query
    $test_query = $wpdb->prepare("
        SELECT 
            SUM(basic_salary + total_allowances + overtime_pay - attendance_penalty) as total_gross,
            SUM(net_salary) as total_net,
            COUNT(id) as processed_count
        FROM {$wpdb->prefix}hrm_payroll 
        WHERE payroll_month = %s
    ", $current_month);
    
    $test_result = $wpdb->get_row($test_query);
    
    if ($wpdb->last_error) {
        echo "<div class='error'>✗ SQL Error: " . $wpdb->last_error . "</div>";
        $issues_found[] = "SQL query error: " . $wpdb->last_error;
    } else {
        echo "<div class='success'>✓ Dashboard query executed successfully</div>";
        echo "<div class='info'>";
        echo "<strong>Query Result:</strong><br>";
        echo "Total Gross: \$" . ($test_result->total_gross ?: 0) . "<br>";
        echo "Total Net: \$" . ($test_result->total_net ?: 0) . "<br>";
        echo "Processed Count: " . ($test_result->processed_count ?: 0);
        echo "</div>";
    }

    // ========================================
    // 9. CHECK NONCE AND PERMISSIONS
    // ========================================
    echo "<h2>9. Security & Permissions</h2>";
    
    if (function_exists('wp_create_nonce')) {
        $test_nonce = wp_create_nonce('hrm_ajax_nonce');
        echo "<div class='success'>✓ Nonce creation working: <code>$test_nonce</code></div>";
    } else {
        echo "<div class='error'>✗ wp_create_nonce function not available</div>";
    }

    if (current_user_can('manage_options')) {
        echo "<div class='success'>✓ Current user has admin permissions</div>";
    } else {
        echo "<div class='warning'>⚠ Current user lacks admin permissions</div>";
    }

    // ========================================
    // 10. SUMMARY AND RECOMMENDATIONS
    // ========================================
    echo "<h2>10. Summary & Recommendations</h2>";
    
    if (empty($issues_found)) {
        echo "<div class='success'>";
        echo "<h3>✓ All Checks Passed!</h3>";
        echo "<p>The payroll module structure is intact. If data is not showing:</p>";
        echo "<ul>";
        echo "<li>Ensure salary structures are defined for employees</li>";
        echo "<li>Generate payroll records using the 'Generate Payroll' tab</li>";
        echo "<li>Check browser console for JavaScript errors</li>";
        echo "</ul>";
        echo "</div>";
    } else {
        echo "<div class='error'>";
        echo "<h3>⚠ Issues Found:</h3>";
        echo "<ol>";
        foreach ($issues_found as $issue) {
            echo "<li>$issue</li>";
        }
        echo "</ol>";
        echo "</div>";
        
        echo "<div class='info'>";
        echo "<h3>🔧 Recommended Fixes:</h3>";
        echo "<ol>";
        
        if (in_array("No payroll data exists", $issues_found)) {
            echo "<li><strong>Generate Payroll Data:</strong> Go to Payroll → Generate Payroll tab and run payroll for employees</li>";
        }
        
        if (in_array("No salary structures configured", $issues_found)) {
            echo "<li><strong>Configure Salary Structures:</strong> Go to Payroll → Salary Structures and set up salary for each employee</li>";
        }
        
        if (strpos(implode(',', $issues_found), 'Missing table') !== false) {
            echo "<li><strong>Recreate Database Tables:</strong> Deactivate and reactivate the plugin to trigger table creation</li>";
        }
        
        if (strpos(implode(',', $issues_found), 'AJAX handler') !== false) {
            echo "<li><strong>Check Plugin Initialization:</strong> Ensure HRM_Payroll::init() is called in the main plugin file</li>";
        }
        
        echo "</ol>";
        echo "</div>";
    }

    // ========================================
    // 11. QUICK FIX ACTIONS
    // ========================================
    echo "<h2>11. Quick Fix Actions</h2>";
    
    if (isset($_GET['action'])) {
        if ($_GET['action'] === 'recreate_tables' && isset($_GET['confirm'])) {
            echo "<div class='info'>Recreating database tables...</div>";
            if (class_exists('HRM_Setup')) {
                HRM_Setup::activate_plugin();
                echo "<div class='success'>✓ Database tables recreated successfully!</div>";
                $fixes_applied[] = "Database tables recreated";
            }
        }
    }
    
    echo "<div style='margin: 20px 0; padding: 20px; background: #f8f9fa; border-radius: 4px;'>";
    echo "<h3>Available Actions:</h3>";
    echo "<p><a href='?action=recreate_tables&confirm=1' style='display: inline-block; padding: 10px 20px; background: #3498db; color: white; text-decoration: none; border-radius: 4px;' onclick='return confirm(\"This will recreate all database tables. Continue?\")'>🔄 Recreate Database Tables</a></p>";
    echo "<p><a href='admin.php?page=hrm-payroll&tab=salary_structures' style='display: inline-block; padding: 10px 20px; background: #27ae60; color: white; text-decoration: none; border-radius: 4px;'>⚙️ Configure Salary Structures</a></p>";
    echo "<p><a href='admin.php?page=hrm-payroll&tab=generate' style='display: inline-block; padding: 10px 20px; background: #e67e22; color: white; text-decoration: none; border-radius: 4px;'>🚀 Generate Payroll</a></p>";
    echo "</div>";

    ?>

    <hr style="margin: 40px 0;">
    <p style="text-align: center; color: #7f8c8d;">
        <strong>Diagnostic Complete</strong> | 
        <a href="<?php echo admin_url('admin.php?page=hrm-payroll'); ?>">Go to Payroll Module</a> | 
        <a href="javascript:location.reload()">Refresh Diagnostics</a>
    </p>
</div>
</body>
</html>
